const express = require('express');
const cors = require('cors');
const NodeMediaServer = require('node-media-server');
const config = require('./config');
const StreamManager = require('./services/streamManager');

const app = express();
app.use(cors());
app.use(express.json());
app.use(express.static('public')); // Serve static files from public directory

// Serve HLS files from media directory
const path = require('path');
const fs = require('fs');
const mediaRoot = config.mediaServer.http.mediaroot || './media';

// Ensure media directory exists
if (!fs.existsSync(mediaRoot)) {
  fs.mkdirSync(mediaRoot, { recursive: true });
}
if (!fs.existsSync(path.join(mediaRoot, 'live'))) {
  fs.mkdirSync(path.join(mediaRoot, 'live'), { recursive: true });
}

// Serve HLS files from /live route with custom 404 handler
app.use('/live', (req, res, next) => {
  // Check if this is a request for HLS files
  if (req.path.endsWith('.m3u8') || req.path.endsWith('.ts')) {
    const filePath = path.join(mediaRoot, 'live', req.path);
    
    // Check if file exists
    if (fs.existsSync(filePath)) {
      // File exists, serve it with proper headers
      const ext = path.extname(filePath);
      res.setHeader('Access-Control-Allow-Origin', '*');
      res.setHeader('Access-Control-Allow-Methods', 'GET, HEAD, OPTIONS');
      res.setHeader('Access-Control-Allow-Headers', 'Content-Type');
      
      if (ext === '.m3u8') {
        res.setHeader('Content-Type', 'application/vnd.apple.mpegurl');
      } else if (ext === '.ts') {
        res.setHeader('Content-Type', 'video/mp2t');
      }
      
      // Serve the file
      res.sendFile(path.resolve(filePath));
    } else {
      // File doesn't exist - provide helpful error
      const streamId = req.path.split('/')[1] || 'unknown';
      res.status(404).json({
        error: 'Stream not found',
        message: `HLS file not found for stream: ${streamId}`,
        hint: 'Make sure the stream is started first using POST /api/streams/start',
        checkUrl: `/api/streams/${streamId}/check`,
        streamPath: req.path
      });
    }
  } else {
    // Not an HLS file request, use static middleware
    express.static(path.join(mediaRoot, 'live'))(req, res, next);
  }
});

// Initialize stream manager
const streamManager = new StreamManager(config);

// Initialize Node Media Server for RTSP relay
const nms = new NodeMediaServer(config.mediaServer);

// API Routes

/**
 * Get list of available streams
 */
app.get('/api/streams', (req, res) => {
  try {
    const streams = streamManager.getActiveStreams();
    res.json({
      success: true,
      streams: streams,
      count: streams.length
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * Start a new RTSP stream from Dahua camera
 * POST /api/streams/start
 * Body: { channel: 1, subtype: 0 }
 */
app.post('/api/streams/start', async (req, res) => {
  try {
    const { channel, subtype } = req.body;
    const streamChannel = channel || config.camera.defaultChannel;
    const streamSubtype = subtype !== undefined ? subtype : config.camera.defaultSubtype;
    
    const streamId = `channel_${streamChannel}_subtype_${streamSubtype}`;
    
    // Get host for URL (use request host or default to localhost)
    // Serve HLS from Express on port 3000, not from node-media-server on 8000
    const host = req.get('host')?.split(':')[0] || 'localhost';
    const hlsUrl = `http://${host}:${config.server.port}/live/${streamId}/index.m3u8`;
    
    // Check if stream already exists
    if (streamManager.isStreamActive(streamId)) {
      return res.json({
        success: true,
        message: 'Stream already active',
        streamId: streamId,
        hlsUrl: hlsUrl
      });
    }
    
    // Start the stream
    await streamManager.startStream(streamId, streamChannel, streamSubtype);
    
    res.json({
      success: true,
      message: 'Stream started successfully',
      streamId: streamId,
      hlsUrl: hlsUrl
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * Stop a specific stream
 * POST /api/streams/stop/:streamId
 */
app.post('/api/streams/stop/:streamId', async (req, res) => {
  try {
    const { streamId } = req.params;
    
    if (!streamManager.isStreamActive(streamId)) {
      return res.status(404).json({
        success: false,
        error: 'Stream not found or already stopped'
      });
    }
    
    await streamManager.stopStream(streamId);
    
    res.json({
      success: true,
      message: 'Stream stopped successfully',
      streamId: streamId
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * Get stream status
 * GET /api/streams/:streamId/status
 */
app.get('/api/streams/:streamId/status', (req, res) => {
  try {
    const { streamId } = req.params;
    const isActive = streamManager.isStreamActive(streamId);
    const streamInfo = streamManager.getStreamInfo(streamId);
    
    // Get host for URL
    // Serve HLS from Express on port 3000, not from node-media-server on 8000
    const host = req.get('host')?.split(':')[0] || 'localhost';
    const hlsUrl = isActive ? `http://${host}:${config.server.port}/live/${streamId}/index.m3u8` : null;
    
    res.json({
      success: true,
      streamId: streamId,
      active: isActive,
      info: streamInfo,
      hlsUrl: hlsUrl
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * Health check endpoint
 */
app.get('/api/health', (req, res) => {
  res.json({
    success: true,
    status: 'healthy',
    timestamp: new Date().toISOString(),
    activeStreams: streamManager.getActiveStreams().length
  });
});

/**
 * Get camera configuration
 */
app.get('/api/camera/config', (req, res) => {
  res.json({
    success: true,
    config: {
      host: config.camera.host,
      port: config.camera.port,
      defaultChannel: config.camera.defaultChannel,
      defaultSubtype: config.camera.defaultSubtype
    }
  });
});

/**
 * API Documentation endpoint
 * GET /api/docs
 */
app.get('/api/docs', (req, res) => {
  const host = req.get('host')?.split(':')[0] || 'localhost';
  const baseUrl = `http://${host}:${config.server.port}`;
  
  res.json({
    success: true,
    title: 'CCTV Streaming API Documentation',
    version: '1.0.0',
    baseUrl: baseUrl,
    endpoints: {
      health: {
        method: 'GET',
        path: '/api/health',
        description: 'Check server health status'
      },
      streams: {
        method: 'GET',
        path: '/api/streams',
        description: 'Get list of active streams'
      },
      startStream: {
        method: 'POST',
        path: '/api/streams/start',
        description: 'Start a new video stream',
        body: {
          channel: 'number (optional, default: 1)',
          subtype: 'number (optional, 0=main, 1=sub1, 2=sub2, default: 0)'
        }
      },
      streamStatus: {
        method: 'GET',
        path: '/api/streams/:streamId/status',
        description: 'Get status of a specific stream'
      },
      stopStream: {
        method: 'POST',
        path: '/api/streams/stop/:streamId',
        description: 'Stop a running stream'
      },
      checkStream: {
        method: 'GET',
        path: '/api/streams/:streamId/check',
        description: 'Check if HLS file exists for a stream'
      },
      cameraConfig: {
        method: 'GET',
        path: '/api/camera/config',
        description: 'Get camera configuration'
      }
    },
    streamAccess: {
      hls: {
        format: 'HLS (HTTP Live Streaming)',
        url: `${baseUrl}/live/{streamId}/index.m3u8`,
        description: 'Access HLS stream once started',
        example: `${baseUrl}/live/channel_1_subtype_0/index.m3u8`
      }
    },
    clientLibraries: {
      javascript: 'Use fetch() or axios',
      python: 'Use requests library',
      curl: 'Use curl command-line tool',
      react: 'Use HLS.js with React hooks',
      android: 'Use ExoPlayer or VideoView',
      ios: 'Use AVPlayer with AVFoundation'
    },
    documentation: `${baseUrl}/API_DOCUMENTATION.md`,
    examples: {
      startStream: {
        curl: `curl -X POST ${baseUrl}/api/streams/start -H "Content-Type: application/json" -d '{"channel": 1, "subtype": 0}'`,
        javascript: `fetch('${baseUrl}/api/streams/start', { method: 'POST', headers: {'Content-Type': 'application/json'}, body: JSON.stringify({channel: 1, subtype: 0}) })`
      }
    }
  });
});

/**
 * Check if HLS file exists for a stream
 * GET /api/streams/:streamId/check
 */
app.get('/api/streams/:streamId/check', (req, res) => {
  try {
    const { streamId } = req.params;
    const fs = require('fs');
    const hlsPath = path.join(mediaRoot, 'live', streamId, 'index.m3u8');
    const exists = fs.existsSync(hlsPath);
    const stats = exists ? fs.statSync(hlsPath) : null;
    
    res.json({
      success: true,
      streamId: streamId,
      fileExists: exists,
      filePath: hlsPath,
      fileSize: stats ? stats.size : 0,
      lastModified: stats ? stats.mtime : null,
      hlsUrl: exists ? `http://${req.get('host')?.split(':')[0] || 'localhost'}:${config.server.port}/live/${streamId}/index.m3u8` : null
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Start Node Media Server
nms.on('preConnect', (id, args) => {
  console.log('[NodeEvent on preConnect]', `id=${id} args=${JSON.stringify(args)}`);
});

nms.on('postConnect', (id, args) => {
  console.log('[NodeEvent on postConnect]', `id=${id} args=${JSON.stringify(args)}`);
});

nms.on('prePublish', (id, StreamPath, args) => {
  console.log('[NodeEvent on prePublish]', `id=${id} StreamPath=${StreamPath} args=${JSON.stringify(args)}`);
});

nms.on('postPublish', (id, StreamPath, args) => {
  console.log('[NodeEvent on postPublish]', `id=${id} StreamPath=${StreamPath} args=${JSON.stringify(args)}`);
});

nms.on('prePlay', (id, StreamPath, args) => {
  console.log('[NodeEvent on prePlay]', `id=${id} StreamPath=${StreamPath} args=${JSON.stringify(args)}`);
});

nms.on('postPlay', (id, StreamPath, args) => {
  console.log('[NodeEvent on postPlay]', `id=${id} StreamPath=${StreamPath} args=${JSON.stringify(args)}`);
});

nms.on('doneConnect', (id, args) => {
  console.log('[NodeEvent on doneConnect]', `id=${id} args=${JSON.stringify(args)}`);
});

nms.run();

// Start Express server
const PORT = config.server.port;
const HOST = config.server.host;

app.listen(PORT, HOST, () => {
  console.log(`\n🚀 CCTV Streaming Server running on http://${HOST}:${PORT}`);
  console.log(`📹 Media Server (RTMP/HLS) running on:`);
  console.log(`   - RTMP: rtmp://${HOST}:${config.mediaServer.rtmp.port}`);
  console.log(`   - HLS:  http://${HOST}:${config.mediaServer.http.port}`);
  console.log(`\n📚 API Endpoints:`);
  console.log(`   - GET  /api/health`);
  console.log(`   - GET  /api/docs (API Documentation)`);
  console.log(`   - GET  /api/streams`);
  console.log(`   - POST /api/streams/start`);
  console.log(`   - POST /api/streams/stop/:streamId`);
  console.log(`   - GET  /api/streams/:streamId/status`);
  console.log(`   - GET  /api/camera/config`);
  console.log(`\n🌐 API is accessible to external clients!`);
  console.log(`   Base URL: http://${HOST === '0.0.0.0' ? 'localhost' : HOST}:${PORT}`);
  console.log(`   Documentation: http://${HOST === '0.0.0.0' ? 'localhost' : HOST}:${PORT}/api/docs`);
  console.log(`\n💡 Example: Start stream with POST /api/streams/start`);
  console.log(`   Body: { "channel": 1, "subtype": 0 }\n`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal received: closing HTTP server');
  streamManager.stopAllStreams();
  nms.stop();
  process.exit(0);
});

process.on('SIGINT', () => {
  console.log('SIGINT signal received: closing HTTP server');
  streamManager.stopAllStreams();
  nms.stop();
  process.exit(0);
});

