const ffmpeg = require('fluent-ffmpeg');
const config = require('../config');

class StreamManager {
  constructor(config) {
    this.config = config;
    this.activeStreams = new Map(); // streamId -> { process, channel, subtype, startTime }
    
    // Set FFmpeg path if specified in config or env
    const ffmpegPath = process.env.FFMPEG_PATH || (config.mediaServer.relay && config.mediaServer.relay.ffmpeg);
    if (ffmpegPath && ffmpegPath !== 'ffmpeg') {
      ffmpeg.setFfmpegPath(ffmpegPath);
      console.log(`[StreamManager] Using FFmpeg from: ${ffmpegPath}`);
    }
  }

  /**
   * Build RTSP URL for Dahua camera
   */
  buildRtspUrl(channel, subtype) {
    const { host, port, username, password } = this.config.camera;
    
    let url = `rtsp://`;
    
    // Add authentication if credentials are provided
    if (username && password) {
      url += `${username}:${password}@`;
    }
    
    url += `${host}:${port}/cam/realmonitor?channel=${channel}&subtype=${subtype}`;
    
    return url;
  }

  /**
   * Start streaming from Dahua camera
   */
  async startStream(streamId, channel, subtype) {
    // Check if stream already exists
    if (this.activeStreams.has(streamId)) {
      throw new Error(`Stream ${streamId} is already active`);
    }

    const rtspUrl = this.buildRtspUrl(channel, subtype);
    const fs = require('fs');
    const path = require('path');
    
    // Create media directory if it doesn't exist
    const mediaRoot = this.config.mediaServer.http.mediaroot || './media';
    const streamPath = path.resolve(mediaRoot, 'live', streamId);
    if (!fs.existsSync(streamPath)) {
      fs.mkdirSync(streamPath, { recursive: true });
    }
    
    // Use absolute paths and normalize for FFmpeg (use forward slashes)
    const hlsOutput = path.resolve(streamPath, 'index.m3u8');
    const segmentPattern = path.join(streamPath, 'segment_%03d.ts').replace(/\\/g, '/');
    
    console.log(`[StreamManager] Starting stream: ${streamId}`);
    console.log(`[StreamManager] RTSP URL: ${rtspUrl}`);
    console.log(`[StreamManager] HLS Output: ${hlsOutput}`);
    console.log(`[StreamManager] Segment Pattern: ${segmentPattern}`);

    // FFmpeg command to convert RTSP directly to HLS
    const ffmpegCommand = ffmpeg(rtspUrl)
      .inputOptions([
        '-rtsp_transport', 'tcp' // Use TCP for more reliable connection
      ])
      .outputOptions([
        '-c:v', 'libx264', // Video codec
        '-preset', 'ultrafast', // Encoding preset
        '-tune', 'zerolatency', // Low latency
        '-c:a', 'aac', // Audio codec
        '-f', 'hls', // HLS format
        '-hls_time', '2', // Segment duration in seconds
        '-hls_list_size', '6', // Number of segments in playlist
        '-hls_flags', 'delete_segments', // Delete old segments
        '-hls_segment_filename', segmentPattern, // Segment filename pattern (forward slashes for FFmpeg)
        '-reconnect', '1', // Reconnect on error
        '-reconnect_at_eof', '1',
        '-reconnect_streamed', '1',
        '-reconnect_delay_max', '2'
      ])
      .output(hlsOutput)
      .on('start', (commandLine) => {
        console.log(`[StreamManager] FFmpeg command: ${commandLine}`);
      })
      .on('stderr', (stderrLine) => {
        // FFmpeg outputs everything to stderr, even normal operation
        // Log important messages
        if (stderrLine.includes('error') || stderrLine.includes('Error') || stderrLine.includes('ERROR') || 
            stderrLine.includes('Connection refused') || stderrLine.includes('Unable to') ||
            stderrLine.includes('Invalid data') || stderrLine.includes('No such file')) {
          console.error(`[StreamManager] FFmpeg error for ${streamId}:`, stderrLine);
        }
        // Log connection and file creation messages
        if (stderrLine.includes('Opening') || stderrLine.includes('Stream #') || 
            stderrLine.includes('Output #') || stderrLine.includes('frame=') ||
            stderrLine.includes('Opening \'') || stderrLine.includes('muxing overhead')) {
          console.log(`[StreamManager] FFmpeg: ${stderrLine.trim()}`);
        }
      })
      .on('error', (err, stdout, stderr) => {
        console.error(`[StreamManager] FFmpeg error for ${streamId}:`, err.message);
        console.error(`[StreamManager] Error code:`, err.code);
        if (stdout) {
          console.error(`[StreamManager] FFmpeg stdout:`, stdout);
        }
        if (stderr) {
          console.error(`[StreamManager] FFmpeg stderr:`, stderr);
        }
        this.activeStreams.delete(streamId);
      })
      .on('end', () => {
        console.log(`[StreamManager] Stream ${streamId} ended`);
        this.activeStreams.delete(streamId);
      })
      .on('progress', (progress) => {
        if (progress && progress.frames) {
          console.log(`[StreamManager] ${streamId} progress: ${progress.frames} frames`);
        }
      });

    // Start the FFmpeg process
    const ffmpegProcess = ffmpegCommand.run();

    // Store stream information
    this.activeStreams.set(streamId, {
      process: ffmpegProcess,
      channel: channel,
      subtype: subtype,
      startTime: new Date(),
      rtspUrl: rtspUrl,
      hlsPath: hlsOutput
    });

    // Wait a bit to check if stream started successfully
    await new Promise((resolve) => setTimeout(resolve, 3000));

    // Check if process is still running
    if (ffmpegProcess.killed) {
      this.activeStreams.delete(streamId);
      throw new Error(`Failed to start stream ${streamId}. Check camera connection and credentials.`);
    }

    // Check if HLS file is being created (wait a bit more for first segment)
    const maxWait = 10000; // 10 seconds max wait
    const checkInterval = 1000; // Check every second
    let waited = 0;
    let fileExists = false;
    
    while (waited < maxWait && !fileExists) {
      await new Promise((resolve) => setTimeout(resolve, checkInterval));
      waited += checkInterval;
      
      if (fs.existsSync(hlsOutput)) {
        fileExists = true;
        console.log(`[StreamManager] HLS file created: ${hlsOutput}`);
        break;
      }
      
      // Check if process died
      if (ffmpegProcess.killed) {
        this.activeStreams.delete(streamId);
        throw new Error(`Stream ${streamId} process died. Check camera connection, RTSP URL, and FFmpeg logs.`);
      }
    }
    
    if (!fileExists) {
      console.warn(`[StreamManager] Warning: HLS file not created after ${waited}ms. Stream may still be initializing.`);
    }

    return streamId;
  }

  /**
   * Stop a specific stream
   */
  async stopStream(streamId) {
    if (!this.activeStreams.has(streamId)) {
      throw new Error(`Stream ${streamId} is not active`);
    }

    const streamInfo = this.activeStreams.get(streamId);
    console.log(`[StreamManager] Stopping stream: ${streamId}`);

    // Kill the FFmpeg process
    if (streamInfo.process && !streamInfo.process.killed) {
      streamInfo.process.kill('SIGKILL');
    }

    this.activeStreams.delete(streamId);
    console.log(`[StreamManager] Stream ${streamId} stopped`);
  }

  /**
   * Stop all active streams
   */
  async stopAllStreams() {
    const streamIds = Array.from(this.activeStreams.keys());
    console.log(`[StreamManager] Stopping all streams: ${streamIds.length}`);
    
    for (const streamId of streamIds) {
      await this.stopStream(streamId);
    }
  }

  /**
   * Check if a stream is active
   */
  isStreamActive(streamId) {
    if (!this.activeStreams.has(streamId)) {
      return false;
    }

    const streamInfo = this.activeStreams.get(streamId);
    
    // Check if process is still running
    if (streamInfo.process && streamInfo.process.killed) {
      this.activeStreams.delete(streamId);
      return false;
    }

    return true;
  }

  /**
   * Get active streams list
   */
  getActiveStreams() {
    // Clean up killed processes
    for (const [streamId, streamInfo] of this.activeStreams.entries()) {
      if (streamInfo.process && streamInfo.process.killed) {
        this.activeStreams.delete(streamId);
      }
    }

    return Array.from(this.activeStreams.keys());
  }

  /**
   * Get stream information
   */
  getStreamInfo(streamId) {
    if (!this.activeStreams.has(streamId)) {
      return null;
    }

    const streamInfo = this.activeStreams.get(streamId);
    const uptime = Math.floor((new Date() - streamInfo.startTime) / 1000); // seconds

    return {
      streamId: streamId,
      channel: streamInfo.channel,
      subtype: streamInfo.subtype,
      rtspUrl: streamInfo.rtspUrl,
      hlsPath: streamInfo.hlsPath,
      startTime: streamInfo.startTime.toISOString(),
      uptime: uptime,
      active: !streamInfo.process.killed
    };
  }
}

module.exports = StreamManager;

